/*
 *  Copyright 2013-2050 Emmanuel BRUN (contact@amapj.fr)
 * 
 *  This file is part of AmapJ.
 *  
 *  AmapJ is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.

 *  AmapJ is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with AmapJ.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * 
 */
 package fr.amapj.view.views.common.gapviewer;

import java.text.SimpleDateFormat;
import java.util.Date;

import com.vaadin.server.FontAwesome;
import com.vaadin.ui.Alignment;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.Notification;
import com.vaadin.ui.VerticalLayout;

import fr.amapj.common.DateUtils;
import fr.amapj.common.GenericUtils.Ret;
import fr.amapj.common.periode.TypPeriode;
import fr.amapj.view.engine.popup.PopupListener;
import fr.amapj.view.engine.tools.BaseUiTools;


/**
 * Permet de créer un bloc permettant de naviguer dans de date en date 
 */
public class  DatePerDateViewer implements AbstractGapViewer
{
	private PopupListener listener;
	
	private DateProvider dateProvider;
	
	private Label titre;
	
	// N'est jamais null 
	private Date currentDate;
	
	
	/**
	 * 
	 */
	public DatePerDateViewer(DateProvider dateProvider,PopupListener listener)
	{
		this.dateProvider = dateProvider;
		this.listener = listener;
				
		Date now = DateUtils.getDateWithNoTime();
		Ret<Date> nextDate = dateProvider.provideDate(now, true);
		if (nextDate.isOK())
		{
			currentDate = nextDate.get();
		}
		else
		{
			currentDate = now;
		}
	}
	
	
	/**
	 * 
	 */
	public VerticalLayout getComponent()
	{
		VerticalLayout vl = new VerticalLayout();
		vl.setMargin(false);
		vl.setSpacing(false);
		vl.addStyleName("date-viewer");
		
		// Bandeau avec les boutons droit / gauche 
		HorizontalLayout hl1 = new HorizontalLayout();
		hl1.setWidth("100%");
		
		
		addButton(false,hl1,e->reculer());
		
		Label empty = new Label();
		hl1.addComponent(empty);
		hl1.setExpandRatio(empty, 1.0f);
		
		
		addButton(true,hl1,e->avancer());
			
		vl.addComponent(hl1);
		
		// Bandeau avec le délai de la date visualisée
		titre = new Label();
		hl1.addStyleName("titre");
		titre.setSizeUndefined();
		vl.addComponent(titre);
		vl.setComponentAlignment(titre, Alignment.MIDDLE_CENTER);
		
		updateTitreValue();
		
		
		return vl;
		
	}

	private void avancer()
	{
		Ret<Date> d = dateProvider.provideDate(DateUtils.addDays(currentDate, 1),true);
		if (d.isOK()==false)
		{
			Notification.show("Impossible", d.msg(), Notification.Type.HUMANIZED_MESSAGE);
			return ;
		}
		
		currentDate = d.get();
		updateTitreValue();
		listener.onPopupClose();  
	}



	private void reculer()
	{
		Ret<Date> d = dateProvider.provideDate(currentDate,false);
		if (d.isOK()==false)
		{
			Notification.show("Impossible", d.msg(), Notification.Type.HUMANIZED_MESSAGE);
			return ;
		}
		
		currentDate = d.get();
		updateTitreValue();
		listener.onPopupClose();  
	}

	
	
	private Button addButton(boolean toRight,HorizontalLayout layout, ClickListener listener)
	{
		String str;
		if (toRight)
		{
			str = "SUIVANT";
		}
		else
		{
			str = "PRECEDENT";
		}
				
		
		Button b = new Button(str);
		b.addClickListener(listener);
	
		
		if (toRight)
		{
			b.setIcon(FontAwesome.ANGLE_DOUBLE_RIGHT);
			b.addStyleName("icon-align-right");
			b.addStyleName("large");
		}
		else
		{
			b.setIcon(FontAwesome.ANGLE_DOUBLE_LEFT);
			b.addStyleName("large");
		}
		
		b.addStyleName("fleche");
		
		layout.addComponent(b);
		
		return b;
		
	}
	
	private void updateTitreValue()
	{
		String toDisplay = getTitreValue();
		titre.setValue(toDisplay);	
	}


	private String getTitreValue() 
	{
		int delay = DateUtils.getDeltaDay(DateUtils.getDateWithNoTime(), currentDate);
		
		if (delay==-1)
		{
			return "hier";
		}		
		if (delay==0)
		{
			return "aujourd'hui";
		}
		if (delay==1)
		{
			return "demain";
		}
		
		if (delay<0)
		{
			if (delay<-60)
			{
				return "il y a "+(-delay/30)+" mois";
			}
			return "il y a "+(-delay)+" jours";
		}
		else
		{
			if (delay>60)
			{
				return "dans "+(delay/30)+" mois";
			}
			return "dans "+delay+" jours";
		}
	}


	@Override
	public Date getDateDebut() 
	{
		return currentDate;
	}


	@Override
	public Date getDateFin() 
	{
		return currentDate;
	}

	@Override
	public TypPeriode getTypPeriode() 
	{
		return TypPeriode.JOUR;
	}



	public VerticalLayout getAdditionnalInfoIfEmpty() 
	{
		SimpleDateFormat dfx = BaseUiTools.isWidthBelow(480) ? new SimpleDateFormat("dd/MM/yy") : new SimpleDateFormat("EEEEE dd MMMMM yyyy");
		
		VerticalLayout vl = new VerticalLayout();
		
		Label label = new Label(dfx.format(currentDate));
		label.addStyleName("titre");
		label.setSizeUndefined();
		vl.addComponent(label);
		vl.setComponentAlignment(label, Alignment.MIDDLE_CENTER);
		
		return vl;
	}
}
